/**
  ******************************************************************************
  * @file    startup_bl.c 
  * @author  Ruediger R. Asche
  * @version V1.0.0
  * @date    14-July-2016
  * @brief   Entry point, fault handlers and IVT for the boot loader
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, THE AUTHOR SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  ******************************************************************************  
  */ 

#include "bl.h"

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Reserve space for the system stack. This is the stack that the startup code runs on until the application entry point is
// called. When the boot loader is in standalone mode, the interrupt handler that processes the host communication also uses
// this stack. Make sure that the stack never overflows!
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */

/* ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// The following are variables created by the linker, indicating where the
// "data" and "bss" segments reside in memory.  The initializers for the
// "data" segment resides immediately following the "text" segment.
//
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////// */
extern unsigned long __ASYST_DATA_LOAD;
extern unsigned long _data;
extern unsigned long _edata;
extern unsigned long _bss;
extern unsigned long _ebss;
extern unsigned long _ramend;

extern void BoardInit(void);

/** @brief NMI Interrupt Handler
 *
 *  @return Does not return
 */

SECTION_FLASH_CODE_BL static void NmiISR_BL(void)
{
    for(;;){}
}

/** @brief Fault Interrupt Handler
 *
 *  This is the hard fault handler. Since we do not enable bus, memory and usage faults,
 *  all those faults will be escalated to this handler (otherwise they would end up in the
 *  default handler because that is where their vector entries point to).
 *
 *  @return Does not return
 */

SECTION_FLASH_CODE_BL static void FaultISR_BL(void)
{
    for (;;){}
}

/** @brief Default Handler for all interrupts not handled otherwise
 *
 *  @return Does not return
 */

SECTION_FLASH_CODE_BL static  void IntDefaultHandler_BL(void)
{
    for (;;){}
}

/** @brief Reset Interrupt Handler
 *
 *  @return Does not return unless main() returns
 */

SECTION_FLASH_CODE_BL static void ResetISR_BL(void)
{
    unsigned    long    *a_Src,*a_Dst;
    // Copy the data segment initializers from flash to SRAM.
    a_Src = &_sidata;
    for(a_Dst = &_data; a_Dst < &_edata; )
    {
        *a_Dst++ = *a_Src++;
    }
    // Zero fill the bss segment.
    for(a_Dst = &_bss; a_Dst < &_ebss; )
    {
        *a_Dst++ = 0;
    }
    SystemInit();
    BoardInit();

    // Call the application's entry point. An application will frequently call the startup entry point of the C runtime
    // initialization, ie __start which will eventually call main(). But we are in the boot loader here, so we can not
    // afford to have additional code that steals code space and does things we do not have control over.
    main_BL();
}

SECTION_FLASH_ISRVECTORS_BL
void (* const g_pfnVectorsBL[IVTENTRYCT])(void) =
{
    // this layout of the IVT corresponds to the STM32F4xx family of controllers.
    (void (*)(void))&_ramend,                                      // initial stack pointer          @0x0000_0000
    ResetISR_BL,                                                  // reset handler                  @0x0000_0004
    NmiISR_BL,                                                     // NMI handler                    @0x0000_0008
    FaultISR_BL,                                                  // hard fault handler             @0x0000_000C
    IntDefaultHandler_BL,                                         // MPU fault handler              @0x0000_0010
    IntDefaultHandler_BL,                                         // bus fault handler              @0x0000_0014
    IntDefaultHandler_BL,                                         // usage fault handler            @0x0000_0018
    0,                                                            // Reserved                       @0x0000_001C
    0,                                                            // Reserved                       @0x0000_0020
    0,                                                            // Reserved                       @0x0000_0024
    0,                                                            // Reserved                       @0x0000_0028
    IntDefaultHandler_BL,                                         // SVCall handler                 @0x0000_002C
    IntDefaultHandler_BL,                                         // Debug monitor handler          @0x0000_0030
    0,                                                            // Reserved                       @0x0000_0034
    IntDefaultHandler_BL,                                         // PendSV handler                 @0x0000_0038
    IntDefaultHandler_BL,                                         // SysTick handler                @0x0000_003C
    /* External Interrupts */
    IntDefaultHandler_BL,                                         // WWDG_IRQHandler                @0x0000_0040                                    
    IntDefaultHandler_BL,                                         // PVD_IRQHandler                 @0x0000_0044            
    IntDefaultHandler_BL,                                         // TAMP_STAMP_IRQHandler          @0x0000_0048            
    IntDefaultHandler_BL,                                         // RTC_WKUP_IRQHandler            @0x0000_004C           
    IntDefaultHandler_BL,                                         // FLASH_IRQHandler               @0x0000_0050                           
    IntDefaultHandler_BL,                                         // RCC_IRQHandler                 @0x0000_0054                             
    IntDefaultHandler_BL,                                         // EXTI0_IRQHandler               @0x0000_0058         
    IntDefaultHandler_BL,                                         //                                @0x0000_005C
    IntDefaultHandler_BL,                                         // EXTI2_IRQHandler               @0x0000_0060           
    IntDefaultHandler_BL,                                         // EXTI3_IRQHandler               @0x0000_0064           
    IntDefaultHandler_BL,                                         // EXTI4_IRQHandler               @0x0000_0068           
    IntDefaultHandler_BL,                                         // DMA1_Stream0_IRQHandler        @0x0000_006C   
    IntDefaultHandler_BL,                                         // DMA1_Stream1_IRQHandler        @0x0000_0070    
    IntDefaultHandler_BL,                                         // DMA1_Stream2_IRQHandler        @0x0000_0074    
    IntDefaultHandler_BL,                                         // DMA1_Stream3_IRQHandler        @0x0000_0078    
    IntDefaultHandler_BL,                                         // DMA1_Stream4_IRQHandler        @0x0000_007C    
    IntDefaultHandler_BL,                                         // DMA1_Stream5_IRQHandler        @0x0000_0080    
    IntDefaultHandler_BL,                                         // DMA1_Stream6_IRQHandler        @0x0000_0084    
    IntDefaultHandler_BL,                                         // ADC_IRQHandler                 @0x0000_0088    
    IntDefaultHandler_BL,                                         // CAN1_TX_IRQHandler             @0x0000_008C          
    IntDefaultHandler_BL,                                         // CAN1_RX0_IRQHandler            @0x0000_0090           
    IntDefaultHandler_BL,                                         // CAN1_RX1_IRQHandler            @0x0000_0094           
    IntDefaultHandler_BL,                                         // CAN1_SCE_IRQHandler            @0x0000_0098           
    IntDefaultHandler_BL,                                         // EXTI9_5_IRQHandler             @0x0000_009C           
    IntDefaultHandler_BL,                                         // TIM1_BRK_TIM9_IRQHandler       @0x0000_00A0 
    IntDefaultHandler_BL,                                         // TIM1_UP_TIM10_IRQHandler       @0x0000_00A4 
    IntDefaultHandler_BL,                                         // TIM1_TRG_COM_TIM11_IRQHandler  @0x0000_00A8 
    IntDefaultHandler_BL,                                         // TIM1_CC_IRQHandler             @0x0000_00AC           
    IntDefaultHandler_BL,                                         // TIM2_IRQHandler                @0x0000_00B0    
    IntDefaultHandler_BL,                                         // TIM3_IRQHandler                @0x0000_00B4    
    IntDefaultHandler_BL,                                         // TIM4_IRQHandler                @0x0000_00B8    
    IntDefaultHandler_BL,                                         // I2C1_EV_IRQHandler             @0x0000_00BC           
    IntDefaultHandler_BL,                                         // I2C1_ER_IRQHandler             @0x0000_00C0           
    IntDefaultHandler_BL,                                         // I2C2_EV_IRQHandler             @0x0000_00C4           
    IntDefaultHandler_BL,                                         // I2C2_ER_IRQHandler             @0x0000_00C8             
    IntDefaultHandler_BL,                                         // SPI1_IRQHandler                @0x0000_00CC    
    IntDefaultHandler_BL,                                         // SPI2_IRQHandler                @0x0000_00D0    
    IntDefaultHandler_BL,                                         // USART1_IRQHandler              @0x0000_00D4    
    IntDefaultHandler_BL,                                         // USART2_IRQHandler              @0x0000_00D8    
    IntDefaultHandler_BL,                                         // USART3_IRQHandler              @0x0000_00DC    
    IntDefaultHandler_BL,                                         // EXTI15_10_IRQHandler           @0x0000_00E0           
    IntDefaultHandler_BL,                                         // RTC_Alarm_IRQHandler           @0x0000_00E4           
    IntDefaultHandler_BL,                                         // OTG_FS_WKUP_IRQHandler         @0x0000_00E8               
    IntDefaultHandler_BL,                                         // TIM8_BRK_TIM12_IRQHandler      @0x0000_00EC 
    IntDefaultHandler_BL,                                         // TIM8_UP_TIM13_IRQHandler       @0x0000_00F0 
    IntDefaultHandler_BL,                                         // TIM8_TRG_COM_TIM14_IRQHandler  @0x0000_00F4 
    IntDefaultHandler_BL,                                         // TIM8_CC_IRQHandler             @0x0000_00F8           
    IntDefaultHandler_BL,                                         // DMA1_Stream7_IRQHandler        @0x0000_00FC           
    IntDefaultHandler_BL,                                         // FSMC_IRQHandler                @0x0000_0100    
    IntDefaultHandler_BL,                                         // SDIO_IRQHandler                @0x0000_0104    
    IntDefaultHandler_BL,                                         // TIM5_IRQHandler                @0x0000_0108    
    IntDefaultHandler_BL,                                         // SPI3_IRQHandler                @0x0000_010C    
    IntDefaultHandler_BL,                                         // UART4_IRQHandler               @0x0000_0110    
    IntDefaultHandler_BL,                                         // UART5_IRQHandler               @0x0000_0114    
    IntDefaultHandler_BL,                                         // TIM6_DAC_IRQHandler            @0x0000_0118       
    IntDefaultHandler_BL,                                         // TIM7_IRQHandler                @0x0000_011C 
    IntDefaultHandler_BL,                                         // DMA2_Stream0_IRQHandler        @0x0000_0120    
    IntDefaultHandler_BL,                                         // DMA2_Stream1_IRQHandler        @0x0000_0124    
    IntDefaultHandler_BL,                                         // DMA2_Stream2_IRQHandler        @0x0000_0128    
    IntDefaultHandler_BL,                                         // DMA2_Stream3_IRQHandler        @0x0000_012C    
    IntDefaultHandler_BL,                                         // DMA2_Stream4_IRQHandler        @0x0000_0130    
    IntDefaultHandler_BL,                                         // ETH_IRQHandler                 @0x0000_0134    
    IntDefaultHandler_BL,                                         // ETH_WKUP_IRQHandler            @0x0000_0138           
    IntDefaultHandler_BL,                                         // CAN2_TX_IRQHandler             @0x0000_013C           
    IntDefaultHandler_BL,                                         // CAN2_RX0_IRQHandler            @0x0000_0140           
    IntDefaultHandler_BL,                                         // CAN2_RX1_IRQHandler            @0x0000_0144           
    IntDefaultHandler_BL,                                         // CAN2_SCE_IRQHandler            @0x0000_0148           
    IntDefaultHandler_BL,                                         // OTG_FS_IRQHandler              @0x0000_014C    
    IntDefaultHandler_BL,                                         // DMA2_Stream5_IRQHandler        @0x0000_0150    
    IntDefaultHandler_BL,                                         // DMA2_Stream6_IRQHandler        @0x0000_0154    
    IntDefaultHandler_BL,                                         // DMA2_Stream7_IRQHandler        @0x0000_0158    
    IntDefaultHandler_BL,                                         // USART6_IRQHandler              @0x0000_015C     
    IntDefaultHandler_BL,                                         // I2C3_EV_IRQHandler             @0x0000_0160           
    IntDefaultHandler_BL,                                         // I2C3_ER_IRQHandler             @0x0000_0164           
    IntDefaultHandler_BL,                                         // OTG_HS_EP1_OUT_IRQHandler      @0x0000_0168    
    IntDefaultHandler_BL,                                         // OTG_HS_EP1_IN_IRQHandler       @0x0000_016C    
    IntDefaultHandler_BL,                                         // OTG_HS_WKUP_IRQHandler         @0x0000_0170            
    IntDefaultHandler_BL,                                         // OTG_HS_IRQHandler              @0x0000_0174    
    IntDefaultHandler_BL,                                         // DCMI_IRQHandler                @0x0000_0178    
    IntDefaultHandler_BL,                                         // CRYP_IRQHandler                @0x0000_017C    
    IntDefaultHandler_BL,                                         // HASH_RNG_IRQHandler            @0x0000_0180 
    IntDefaultHandler_BL,                                         // FPU_IRQHandler                 @0x0000_0184    
    IntDefaultHandler_BL,                                         // UART7_IRQHandler               @0x0000_0188           
    IntDefaultHandler_BL,                                         // UART8_IRQHandler               @0x0000_018C    
    IntDefaultHandler_BL,                                         // SPI4_IRQHandler                @0x0000_0190    
    IntDefaultHandler_BL,                                         // SPI5_IRQHandler                @0x0000_0194            
    IntDefaultHandler_BL,                                         // SPI6_IRQHandler                @0x0000_0198    
    IntDefaultHandler_BL,                                         // SAI1_IRQHandler                @0x0000_019C    
    IntDefaultHandler_BL,                                         // LCD_TFTIRQHandler              @0x0000_01A0    
    IntDefaultHandler_BL,                                         // LCD_TFTIRQErrorHandler         @0x0000_01A4 
    IntDefaultHandler_BL,                                         // DMA2D_IRQHandler               @0x0000_01A8          
};    
    
SECTION_FLASH_ISRVECTORS_FW
void (* const g_pfnVectorsFWStubs[IVTENTRYCT])(void) = {0};  // this has a NOLOAD directive in the lcf, so it won't be part of the image!



